/* 
 * Poisson noise GIMP plugin
 * 
 * noise.c
 * Copyright 2008 by Marco Rossini
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2+
 * as published by the Free Software Foundation.
 */

#include "plugin.h"

guint poisson(float lambda, guint n)
{
	double L, p;
	guint k;

	if (n == 0)
		return 0;

	L = exp(-lambda);
	k = 0;
	p = 1;
	do {
		k++;
		p = p * rand() / (double) RAND_MAX;
	} while (p >= L && k <= 255);
	return k - 1 + poisson(lambda, n - 1);
}

void
noise_gen (GimpDrawable * drawable, GimpPreview * preview)
{
  GimpPixelRgn rgn_in, rgn_out;
  gint i, x1, y1, x2, y2, width, height, x, c;
  guchar *line;
  float val[4];

  if (preview)
    {
      gimp_preview_get_position (preview, &x1, &y1);
      gimp_preview_get_size (preview, &width, &height);
      x2 = x1 + width;
      y2 = y1 + height;
    }
  else
    {
      gimp_drawable_mask_bounds (drawable->drawable_id, &x1, &y1, &x2, &y2);
      width = x2 - x1;
      height = y2 - y1;
    }

  gimp_pixel_rgn_init (&rgn_in, drawable, x1, y1, width, height, FALSE,
		       FALSE);
  gimp_pixel_rgn_init (&rgn_out, drawable, x1, y1, width, height,
		       preview == NULL, TRUE);

  /* cache some tiles to make reading/writing faster */
  gimp_tile_cache_ntiles (2 * drawable->width / gimp_tile_width () + 1);

  /* FIXME: replace by GIMP functions */
  line = (guchar *) malloc (channels * width * sizeof (guchar));

  /* read, generate and write */
  if (!preview)
    gimp_progress_init (_("Generating Poisson noise..."));
  for (i = 0; i < y2 - y1; i++)
    {
      if (!preview && i % 10 == 0)
	gimp_progress_update (i / (double) height);
      gimp_pixel_rgn_get_row (&rgn_in, line, x1, i + y1, width);

      if (channels > 2 && settings.luminance_only) {
        for (x = 0; x < width; x++)
	  {
	    for (c = 0; c < channels; c++) {
	      val[c] = (float) line[x * channels + c];
	    }
	    rgb2ycbcr(&(val[0]), &(val[1]), &(val[2]));
	    val[0] = poisson(val[0], settings.photons);
	    val[0] /= (double) settings.photons;
	    ycbcr2rgb(&(val[0]), &(val[1]), &(val[2]));
	    for (c = 0; c < channels; c++) {
	      line[x * channels + c] = (guchar) CLIP(val[c] + 0.5, 0, 255);
	    }
	  }
      } else {
        for (x = 0; x < width; x++)
	  {
	    for (c = 0; c < ((channels % 2) ? channels : channels - 1); c++) {
	      val[c] = (float) line[x * channels + c];
	      val[c] = poisson(val[c], settings.photons);
	      val[c] /= (double) settings.photons;
	      line[x * channels + c] = (guchar) CLIP(val[c] + 0.5, 0, 255);
	    }
	  }
      }
      gimp_pixel_rgn_set_row(&rgn_out, line, x1, i + y1, width);
    }

  /* FIXME: replace by gimp functions */
  free (line);

  if (preview)
    {
      gimp_drawable_preview_draw_region (GIMP_DRAWABLE_PREVIEW (preview),
					 &rgn_out);
      return;
    }
  gimp_drawable_flush (drawable);
  gimp_drawable_merge_shadow (drawable->drawable_id, TRUE);
  gimp_drawable_update (drawable->drawable_id, x1, y1, width, height);
}
